﻿using System;
using System.Windows.Forms;
using Tektronix.LogicAnalyzer.TpiNet;

namespace DataWindowPlugInEx.View
{
    /// <summary>
    /// This is the control that will be shown inside the "System" window.  There are really
    /// no specific restrictions to what it looks like.  For this example, I modeled it after
    /// the built-in data window system window controls (listing, waveform, etc).
    /// </summary>
    /// <remarks>
    /// Note that the size will be restricted to 88x70.
    /// </remarks>
    public partial class SystemWindowControl : UserControl
    {
        private DataWindowPlugInEx PlugIn { get; set; }
        private TlaForm DataWindow { get; set; }

        public SystemWindowControl(DataWindowPlugInEx plugIn)
        {
            InitializeComponent();

            PlugIn = plugIn;
            if (PlugIn != null)
            {
                // PrimaryForm object *should* be sub-classed from TlaForm.
                DataWindow = PlugIn.PrimaryForm as TlaForm;
                if (_textbox != null)
                {
                    _textbox.Text = PlugIn.UserName;
                }
            }
            
            RegisterEvents();
        }

        #region Method(s)

        private void RegisterEvents()
        {
            if (PlugIn != null)
            {
                PlugIn.UserNameChanged += PlugInUserNameChanged;
            }
            if (_button != null)
            {
                _button.Click += ButtonClick;
            }
            if (_textbox != null)
            {
                _textbox.TextChanged += TextboxTextChanged;
            }
        }

        private void UnregisterEvents()
        {
            if (PlugIn != null)
            {
                PlugIn.UserNameChanged -= PlugInUserNameChanged;
            }
            if (_button != null)
            {
                _button.Click -= ButtonClick;
            }
            if (_textbox != null)
            {
                _textbox.TextChanged -= TextboxTextChanged;
            }
        }

        #endregion

        #region Event Handler(s)

        /// <summary>
        /// Method to update the plug-in's username, should it change.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void PlugInUserNameChanged(object sender, EventArgs e)
        {
            if (_disposed) throw new ObjectDisposedException(ToString(), "object has been disposed");
            if (_textbox != null && PlugIn != null)
            {
                _textbox.Text = PlugIn.UserName;
            }
        }

        /// <summary>
        /// Method to activate the DataWindow of this plug-in.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void ButtonClick(object sender, EventArgs e)
        {
            if (_disposed) throw new ObjectDisposedException(ToString(), "object has been disposed");
            if (DataWindow != null)
            {
                DataWindow.Activate();
            }
        }

        /// <summary>
        /// Method to change the username for this plug-in.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void TextboxTextChanged(object sender, EventArgs e)
        {
            if (_disposed) throw new ObjectDisposedException(ToString(), "object has been disposed");
            if (_textbox != null && PlugIn != null)
            {
                PlugIn.UserName = _textbox.Text;
            }
        }

        #endregion
    }
}
